%This code estimates a VAR for monetary policy analysis using data from
%Chile on three variables: inflation, the output gap, and the monetary
%policy rate (MPR), for the period 2001:Q3-2019:Q3.
%It identifies structural shocks using a Cholesky decomposition of the 
%variance-covariance matrix of reduced-form residuals. 
%The code plots impulse-responses to a shock to the MPR, with bootstrapped
%error bands, and computes the forecast error variance decomposition (FEVD). 
%The FEVD of variables 1, 2, and 3 is stored in matrices fevd1, fevd2, and
%fevd3, respectively. In these matrices, rows are the forecast horizon,
%and columns contain the contribution of the shock to each variable to the 
%FEV.
%If you want to alter the specification (number of variables, number of
%lags, etc.), you have to read and adjust the code line by line. It's not
%written so as to be modified by setting a few options.
%The notation is based on the handbook and accompanying
%code by Andrew Blake and Haroon Mumtaz: Applied Bayesian Econometrics for
%Central Bankers. The code uses functions provided by the authors.
%Juan Guerra-Salas
%May 2020

clear
addpath('functions'); %functions written by Blake and Mumtaz
%load data
addpath('data');
data=xlsread('\data\dataChile.xlsx'); %inflation, output gap, MPR. 2001:Q3-2019:Q3 

%Plot variables
tt=2001.5:0.25:2019.5; %dates for x axis; convention: Q1=2010.00, Q2=2010.25, Q3=2010.50, Q4=2010.75
figure(1)
subplot(3,1,1)
plot(tt,data(:,1)); grid on
title('Inflation');
axis tight
subplot(3,1,2)
plot(tt,data(:,2)); grid on
title('Output gap');
axis tight
subplot(3,1,3)
plot(tt,data(:,3)); grid on
title('Monetary policy rate');
axis tight

N=size(data,2); %number of endogenous variables
L=1; %number of lags in the VAR
Y=data; %matrix of endogenous variables
X=[ones(size(Y,1),1) lag0(data,1)]; %matrix of regressors: constant and first lag of variables
Y=Y(L+1:end,:); %adjust sample for observations that are lost due to lags
X=X(L+1:end,:);
T=rows(X); %number of observations (adjusted)

%Estimation of parameters 
beta=inv(X'*X)*(X'*Y); %OLS
%Estimation of variance-covariance matrix of residuals
e=Y-X*beta; %Residuals
Sigma=(e'*e)/(T-(L*N+1)); %Var-cov matrix of residuals corrected for degrees of freedom (L*N+1 is the # of param. per eqn. when including a cons.)
%Covariance matrix of parameters
V_beta=kron(Sigma,inv(X'*X)); %See p. 29 of the handbook, last paragraph

%Cholesky decomposition of variance-covariance matrix of residuals
A0=chol(Sigma);

%Impulse responses
v=zeros(20,N); %matrix that will contain the shock
v(L+1,3)=1; %here is the shock to the MPR
yhat=zeros(20,N); %matrix that will contain the IRFs
for i=2:20 %start in the second row; the first is "lost" because we need to use the first lag
    yhat(i,:)=[0 yhat(i-1,:)]*beta+v(i,:)*A0; %build IRFs by simulating; set cons=0 because IRFs describe deviations from mean
end

%Bootstrap error bands
%See slide 20 in http://pareto.uab.es/lgambetti/VAR_Forecasting.pdf 
Bootrep=10^4; %Number of bootstrap repetitions
yhatboot=zeros(20,N,Bootrep); %Bootstrapped impulse responses
for j=1:Bootrep
    Yboot=zeros(T,N); %Matrix that will contain artificial data
    for jj=2:T
        Yboot(1,:)=X(1,:)*beta+datasample(e,1);
        Yboot(jj,:)=[1 Yboot(jj-1,:)]*beta+datasample(e,1); %artificial data is built by drawing randomly from estimated residuals
    end
    Xboot=[ones(size(Yboot,1),1) lag0(Yboot(:,:),1) ];
    Yboot=Yboot(2:end,:);
    Xboot=Xboot(2:end,:);
    betaboot=inv(Xboot'*Xboot)*(Xboot'*Yboot);
    eboot=Yboot-Xboot*betaboot;
    Sigmaboot=(eboot'*eboot)/(T-(L*N+1));
    A0boot=chol(Sigmaboot);
    vboot=zeros(20,N);
    vboot(L+1,3)=1; %shock to the MPR
    for jjj=2:20 %IRFs with artificial data
        yhatboot(jjj,:,j)=[0 yhatboot(jjj-1,:,j)]*betaboot+vboot(jjj,:)*A0boot;
    end
end

%Normalize IRFs
norm=0; %Set to 1 to obtain normalized IRFs, i.e., shock of size 1
if norm==1
    yhat=yhat/yhat(L+1,3);
    for j=1:Bootrep
       yhatboot(:,:,j)=yhatboot(:,:,j)/yhatboot(L+1,3,j); 
    end
end

%Plot IRFs with 68% error bands
figure(2)
subplot(3,1,1)
plot([yhat(2:end,1) prctile(yhatboot(2:end,1,:),[16],3) prctile(yhatboot(2:end,1,:),[84],3)]); grid on
title('Response of inflation');
axis tight
subplot(3,1,2)
plot([yhat(2:end,2) prctile(yhatboot(2:end,2,:),[16],3) prctile(yhatboot(2:end,2,:),[84],3)]); grid on
title('Response of output gap');
axis tight
subplot(3,1,3)
plot([yhat(2:end,3) prctile(yhatboot(2:end,3,:),[16],3) prctile(yhatboot(2:end,3,:),[84],3)]); grid on
title('Response of monetary policy rate');
xlabel('Quarters')
axis tight
legend('Estimated Response','Lower 16%','Upper 84%');

%Variance decomposition
%Reference and notation: Lutkepohl (2005, section 2.3.3, p. 63).
%Code for FEVD is written to accomodate any number of lags and forecast
%horizons (defined in h), but for a different number of variables, you must
%change the third loop, which computes summations.
P=A0'; %Express Cholesky decomposition of var-cov matrix of residuals as lower triangular (entries above main diag are 0)
h=[1 4 8 12]; %Forecast horizons
fevd1=zeros(size(h,2),N); %Store FEVD of variable 1. Rows denote horizon h; cols denote contribution of each variable.
fevd2=zeros(size(h,2),N); %Store FEVD of variable 2
fevd3=zeros(size(h,2),N); %Store FEVD of variable 3
se_fcast=zeros(size(h,2),N); %Store standard error of forecast
beta_mat=reshape(beta,N*L+1,N); %Matrix of coefficients
A_=(beta_mat(2:end,:))'; %Coeficients, excluding constants (not needed for FEVD), transposed, so each row refers to each equation in system
A=zeros(N*L,N*L); %A will be the matrix of VAR coefficients in companion form, ie VAR(p) expressed as VAR(1)
A(1:N,:)=A_;
Ik=eye(N); %N*N identity matrix
if L>1 %If the model has more than one lag, compute matrix of coefficients in companion form
    for ll=1:(L-1)
        A((N+1)*ll:N*ll*2,1:N)=Ik;
    end
end
J=zeros(N,N*L);
J(:,1:N)=Ik; %See variable J in Lutkepohl. It's just a matrix used for operations
for jj=1:size(h,2) %This loop: for each forecast horizon...
    for nn=1:N %This loop: for each variable...
        MSE=zeros(N,N); % Diagonal elements will be FEV of each variable. Lutkepohl sometimes refers to FEV as mean squared error (MSE)
        Cont1=0;
        Cont2=0;
        Cont3=0;
        for ii=0:(h(jj)-1) %This loop: computes summations as ii ranges from zero to horizon-1
            %MSE matrix
            temp0=J*A^ii*J'*Sigma'*(J*A^ii*J')';
            MSE=MSE+temp0; 
            Theta=J*A^ii*J'*P;
            %Contribution of shock to variable 1
            temp1=(Ik(:,1)'*Theta*Ik(:,nn))^2;
            Cont1=Cont1+temp1;
            %Contribution of shock to variable 2
            temp2=(Ik(:,2)'*Theta*Ik(:,nn))^2;
            Cont2=Cont2+temp2;
            %Contribution of shock to variable 3
            temp3=(Ik(:,3)'*Theta*Ik(:,nn))^2;
            Cont3=Cont3+temp3;
        end
        fevd1(jj,nn)=Cont1/MSE(1,1)*100;
        fevd2(jj,nn)=Cont2/MSE(2,2)*100;
        fevd3(jj,nn)=Cont3/MSE(3,3)*100;
        se_fcast(jj,:)=sqrt(diag(MSE));
    end
end